/*---------------------------------------------------------------------------
 Copyright:      Henning Schaper  mailto: henningschaper@gmx.de
 Author:         Henning Schaper
 Remarks:        none
 known Problems: none
 Version:        v1.7    26.06.2014
 Description:  	 Numeric Controll - 
 				 Anzeige für Digitale Messschieber 
 				 Prozessor: Atmel Atmega 644
 				 Features:
 				 - X- und Y-Achse; Auflösung 1/100mm
 				 - Y-Achse *2 für Durchmesser
 				 - Einstellbarer Offset für X- und Y
 				 - GLCD (KS0108)
 				 - Nullung der Anzeige per Taster
 				 - Berechnung und Anzeige der Schnittgeschwindigkeit
 				 - Berechnung und Anzeige des Voschubs
 				 - Drezahlanzeige (Messung über Hallsensor an Hauptspindel)

Compiliert mit Avr-gcc in version 4.8.1.
				 		
----------------------------------------------------------------------------*/

//---------------------- Define's  ----------------------//

//Taster
#define SWITCHES_PORT 	PORTC
#define SWITCHES_PIN 	PINC
#define SWITCHES_DDR 	DDRC

#define SET_CHANGE		PC4
#define SET_ZERO		PC3
#define SET_PLUS		PC2
#define SET_MINUS		PC1

//Messschieber
#define XY_PORT 		PORTD
#define XY_PIN 			PIND
#define XY_DDR 			DDRD

#define X_DATA 			PD4
#define X_ZERO 			PD5
#define Y_DATA 			PD1
#define Y_ZERO 			PD0

//HALLSENSOREN
#define HALL_PORT 		PORTD
#define HALL_PIN 		PIND
#define HALL_DDR 		DDRD

#define HALL_A			PD6
#define HALL_B			PD7

 //Sonstiges
#define PI 			314
#define F_CPU 1600000UL 

//----------------------- Include's  ----------------------//

#include <util/delay.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <math.h> 				 
#include "mylcd.h"
#include "fonts.h"
#include "background.h"
#include "stdlib.h"

//---------------- Funktionsdeklaration  ----------------//

void system_init(void);
int32_t convert_position(uint8_t *);
void write_position(uint8_t , int32_t);
void get_switches(void);
void write_parameters(void);
void draw_cross(uint8_t, uint8_t);
void calc_parameters(void);
void delay_loop (uint8_t);

//---------------------- Variablen  ---------------------//

volatile uint8_t x_value[30];
volatile uint8_t x_index = 0;
volatile int32_t x_position = 0;
volatile int32_t last_x_position = 0;   
volatile uint8_t y_value[30];
volatile uint8_t y_index = 0;
volatile int32_t y_position = 0;
volatile uint8_t x_delay_counter = 0;
volatile uint8_t y_delay_counter = 0;
int16_t x_offset = 0;
int16_t y_offset = 0;
uint8_t Display_Buffer[30];
uint8_t x_y_flag = 0;


//Variablen für Schnittparameter
volatile int32_t counter = 2;
volatile uint16_t frequenzy = 0;
volatile int32_t feed = 0;
uint32_t cutting_speed = 0;
uint8_t diameter = 0;


//------------------------ MAIN  -----------------------//
int main (void)
{	
	system_init();
	
	while(1) 										// Endlosschleife
	{
		get_switches();
		lcd_set_cursor(0,0);
		
		if(x_y_flag==0)								//Checkboxen ankreuzen, für aktive Auswahl
		{
			draw_cross(5,BLACK);
			draw_cross(29,WHITE);
		} 
		
		else 
		{
			draw_cross(5,WHITE);
			draw_cross(29,BLACK);
		}

		_delay_ms(50);								// Wichtiger Delay, sonst entstehen unsinnige Werte!	
		write_position(0,x_position+x_offset);		//Werte aufs Display schreiben		
		write_position(24,2*fabs(y_position)+y_offset);
		calc_parameters();
		write_parameters();	
	}
}

void system_init(void)
{	
	//Timer klarmachen, jede Sekunde ISR
	OCR1A = 15625-1;										//TOP-Value für 1 Sekunde -1 CTC-Delay	
	TCCR1B |= (1 << WGM12) | (1 << CS12 ) | (1 << CS10);  	//Prescaler 1024@16Mhz , CTC (Mode 4)
	TIMSK1 |= (1 << OCIE1A); 								//Interrupt Enable	

	//Timer klarmachen, jede Millisekunde ISR
	OCR0A = 250-1;											//TOP-Value für 1 Millisekunde -1 CTC-Delay
	TCCR0A |= (1 << WGM01);									//CTC (Mode2)
	TCCR0B |= (1 << CS00) | (1 << CS01);					//Prescaler 64@16Mhz
	TIMSK0 |= (1 << OCIE0A);								//Interrupt Enable					 	

	//Interrupts für Messschieber initialisieren 
	EICRA  	|= (1 << ISC11) | (1 << ISC10) | (1 << ISC01) | (1 << ISC00);		// INT0 und INT1 konfigurieren, steigende Flanke
	EIMSK 	|=  (1 << INT1) | (1 << INT0);									// Interrupt für INT0 und INT1 aktivieren
	
	//Ein und Ausgänge
	SWITCHES_DDR 	&= ~((1 << SET_ZERO) | (1 << SET_PLUS) | (1 << SET_MINUS) | (1 << SET_CHANGE)); 	//Taster Eingänge
	SWITCHES_PORT 	|= ((1 << SET_ZERO) | (1 << SET_PLUS) | (1 << SET_MINUS) | (1 << SET_CHANGE));		//PULLUPS


	XY_DDR	&= ~((1 << X_DATA) | (1 << Y_DATA));	//Eingänge für Messschieber 	
	XY_DDR	|= (1 << X_ZERO) | (1 << Y_ZERO);       //Ausgänge für Messchieber	
	XY_PORT |= (1 << X_DATA)| (1 << Y_DATA);		//PULLUPS									
	//XY_PORT	|= (1 << X_ZERO) | (1 << Y_ZERO);		//Ausgänge auf High 	

	//Pin Change Interrupt klarmachen
	PCMSK3 |= (1 << PCINT31) | (1 << PCINT30);
	PCICR |= (1 << PCIE3);

	// GLCD initialisieren
	lcd_init();
	lcd_clear();
	lcd_draw_fullscreen_bmp(backgroundbmp);			//Hintergrundbild zeichnen

	_delay_ms(500);		//etwas warten bevor Interrups aktiviert, aus irgendeinem Grund wichtig!!
	
	sei();				//Interrupts zulassen		
}

//---------- Kleine Warteschleife für Entrellung etc.  -----------//
//------------------------- TESTED: O.K. -------------------------//
void delay_loop (uint8_t t)
{
	for (uint8_t n = 0; n<t ; n++)
	{
		_delay_ms(100);
	}
}

//----------- Aktuellen Positionswerte aufs Display   ------------//
//----------------    schreiben TESTED: O.K.      ----------------//
void write_position(uint8_t y,int32_t position)
{	
	lcd_set_cursor(25,y);
	unsigned char temp;
	
	// "." = Plathalter für Minus-Zeichen (8px) ; "/" = Platzhalter für Ziffer (14px)
	// damit Darstellung Rechtsbündig vor dem Komma ist.
	if (position >= 10000)                //100er Stelle            
    lcd_puts_p(numbersbig,PSTR("."));  
  	else if (position >= 1000)            //10er Stelle            
    lcd_puts_p(numbersbig,PSTR("./"));
  	else if (position >= 0)               //1er Stelle
    lcd_puts_p(numbersbig,PSTR(".//"));
  	else if (position > -100)             //-1er Stelle Sonderfall -0.xx
    lcd_puts_p(numbersbig,PSTR("//-"));
  	else if (position > -1000)            //-1er Stelle
    lcd_puts_p(numbersbig,PSTR("//"));
  	else if (position > -10000)           //-10er Stelle            
    lcd_puts_p(numbersbig,PSTR("/")); 

	itoa((position/100), Display_Buffer, 10 );			//Vorkommastellen aufs Display schreiben
	lcd_puts(numbersbig, Display_Buffer);
	
	temp = fabs(position)-((fabs(position/100))*100);	//Nachkommastellen berechnen und aufs Display schreiben		
	itoa(temp, Display_Buffer, 10 );		
	lcd_set_cursor(83,y);
	if (temp < 10)
	{
		lcd_puts_p(numbersbig,PSTR("0"));				//Sonderfall: x.0x
	}
	lcd_puts(numbersbig, Display_Buffer);
}


//----------------- Schnittparameter berechnen   -----------------//
//----------------         Baustelle!!!!!         ----------------//
void calc_parameters(void)
{
	//Schrittweise Berechnung der Schnittgeschwindigkeit
	diameter = (2*fabs(y_position)+y_offset)/100;
	cutting_speed = diameter*(frequenzy/10);
	cutting_speed = cutting_speed/100;
	cutting_speed = cutting_speed*PI;
	cutting_speed = cutting_speed/100;
}

//----------- Schnittparameter auf Display schreiben   ------------//
//----------------           Tested: O.K.          ----------------//
void write_parameters(void)
{
	uint16_t temp_frequenzy;			//Temporäre Variable für Drehzahl; da "frequenzy" volatile ist und sich
										// stets innerhalb "write_parameters" ändern kann und damit Darstellungs-
										// fehler erzeugen könnte
	temp_frequenzy = frequenzy;
	
	if(temp_frequenzy>9999)				//Wert nach oben terminieren um Darstellungsfehler auf dem Display auszuschließen
	{
		temp_frequenzy=9999;
	}
									
	lcd_set_cursor(9,53);
	if(temp_frequenzy<10)					//Drehzahlausgabe, rechtsbündige Darstellung ":" als Platzhalter		
	{
		lcd_puts_p(numberssmall,PSTR(":::"));
	}
	else if(temp_frequenzy<100)
	{
		lcd_puts_p(numberssmall,PSTR("::"));
	}
	else if(temp_frequenzy<1000)
	{
		lcd_puts_p(numberssmall,PSTR(":"));
	}
	itoa(temp_frequenzy, Display_Buffer, 10 );
	lcd_puts(numberssmall, Display_Buffer);
	
	//Vorschubausgabe, Vorkomma- und Nachkommastelle
	if(feed > 99)				//Wert nach oben terminieren um Darstellungsfehler auf dem Display auszuschließen
	{
		feed = 99;
	}
	else if(feed < 0)
	{
		feed = 0;
	}

	lcd_set_cursor(56,53);					//Vorschub Vorkommastelle
	itoa(feed/10, Display_Buffer, 10 );
	lcd_puts(numberssmall, Display_Buffer);

	lcd_set_cursor(65,53);					//Vorschub Nachkommastelle
	itoa((feed-(feed/10)*10), Display_Buffer, 10 );
	lcd_puts(numberssmall, Display_Buffer);
	
	//Ausgabe der Schnittgeschwindigkeit, rechtsbündige Darstellung ":" als Platzhalter
	lcd_set_cursor(98,53);
	
	if(cutting_speed >= 100)	
	{
		if(cutting_speed>999)	//Wert nach oben terminieren um Darstellungsfehler auf dem Display auszuschließen
		{
			cutting_speed=999; 
		}
	}
	else if (cutting_speed >= 10)
	lcd_puts_p(numberssmall,PSTR(":"));
	else 
	lcd_puts_p(numberssmall,PSTR("::"));
	itoa(cutting_speed, Display_Buffer, 10 );
	lcd_puts(numberssmall, Display_Buffer);
}

//---------------- Auswertung der Taster; TO-DO:  ----------------//
//----------------        Bei +/-                 ----------------//
//---------------- gedrückt = schnelldurchlauf,   ----------------//
//---------------- ansonsten schrittweise         ----------------//
void get_switches(void)
{
	if(!(SWITCHES_PIN & (1<<SET_CHANGE)))
	{
	delay_loop(60);
		x_y_flag = !x_y_flag;
	}
	else if(!(SWITCHES_PIN & (1<<SET_ZERO)))
	{
		delay_loop(4);
		if(x_y_flag==0)
		{
			x_offset=0;
			XY_PORT |= (1 << X_ZERO);
			delay_loop(10);
			XY_PORT &= ~(1 << X_ZERO);
		}

		if(x_y_flag==1) 
		{
			y_offset=0;
			XY_PORT |= (1 << Y_ZERO);
			delay_loop(10);
			XY_PORT &= ~(1 << Y_ZERO);
		}		
	}
	else if(!(SWITCHES_PIN & (1<<SET_PLUS)))
	{
		delay_loop(4);
		if(x_y_flag==0) x_offset++;
		if(x_y_flag==1) y_offset++;
	}
	else if(!(SWITCHES_PIN & (1<<SET_MINUS)))
	{
		delay_loop(4);
		if(x_y_flag==0) x_offset--;
		if(x_y_flag==1) y_offset--;
	}	
}

//-------------------- Interrupt jede Sekunde --------------------//
//----------------          Tested: O.K.          ----------------//
ISR(TIMER1_COMPA_vect)
{	
	frequenzy = counter*30; //Zähler/2 * 60 Sekunden = Upm
	feed = fabs((fabs(x_position)-fabs(last_x_position))/counter/2); //Berechnung des Vorschubs
	last_x_position = x_position;	// aktuelle X-Position speichern für deltaX
	counter = 0; 
}

ISR(TIMER0_COMPA_vect)
{
	x_delay_counter++;
	y_delay_counter++;
}


//----------- Interrupt pro Umdrehung der Hauptspindel ------------//
//----------------          Tested: O.K.          ----------------//
ISR(PCINT3_vect)
{
	counter++; // 2 Inkremente/Umdrehung
}

//---------------- Interrups Service Routinen für ----------------//
//---------------- Daten von Messschiebern; INT0, ----------------//
//---------------- INT1 jeweils Clocksignal, PD1  ----------------//
//---------------- und PD4 jeweils Datensignal    ----------------//
//----------------          Tested: O.K.          ----------------//
ISR(INT1_vect)
{
	if (x_index==0 && x_delay_counter < 100)			//Delay (100ms) zwischen den Datenpaketen abwarten
	{		
		x_delay_counter=0;											
		return;
	}
	else 												//dann erst mit auswertung beginnen
	{
		x_delay_counter=0;

		if (x_index != 24)
		{	
			if (XY_PIN & (1<<X_DATA))					// 1 für Datenbit "high" ansonsten 0 belassen
    		{
      			x_value[x_index] =1;		
    		}
			x_index++;									// Courser auf nächste Stelle für nächsten Interrupt
		}	 	
	
		if (x_index == 24)								// Datenpaket zuende, beginne mit Umwandlung in signed long
		{		
			x_index=0;									//"\r"
			x_position = convert_position(x_value);
		}
	}
}


ISR(INT0_vect)
{
	if (y_index==0 && y_delay_counter < 100)				//Delay (100ms) zwischen den Datenpaketen abwarten
	{
		y_delay_counter=0;
		return;
	}
	else 												//dann erst mit auswertung beginnen
	{
		y_delay_counter=0;

		if (y_index != 24)
		{
			if (XY_PIN & (1<<Y_DATA))					// 1 für Datenbit "high" ansonsten 0 belassen
    		{
      			y_value[y_index] =1;					
    		}
		y_index++;										// Courser auf nächste Stelle für nächsten Interrupt
		}	 	
	
		if (y_index == 24)								// Datenpaket zuende, beginne mit Umwandlung in signed long
		{		
			y_index=0;									//"\r"
			y_position = convert_position(y_value);	
		}
	}
}

//------------- Binäres ARRAY in SignedLONG umwandeln -------------//
//----------------           Tested: O.K.          ----------------//
int32_t convert_position(uint8_t *i)
{
	uint16_t lookup_table[17]= {1,2,4,8,16,32,64,128,256,512,1024,2048,4096,8192,16384,32768,65535}; // 2^n dauert zu lange!
	int32_t position = 0;
	uint8_t n=0;
	
	do
	{												//Beginne mit LSB
		if(i[n]==1)									//Wenn 1								
		{
			position += lookup_table[n];			//dann Entsprechende Wertigkeit des Bits zuaddieren 
			i[n]=0;									//Bit gleich wieder für das nächste Datenpaket löschen
		}
		n++;										//nächste Stelle		
	}
	while(n<17);									// Für alle 17 Datenbits	
	
	if(i[20]==1)									// Vorzeichen setzen
	{
		position=position*(-1);
		i[20]=0;									// Vorzeichenbit für nächstes Datenpaket 0 setzen
	}

	n=0;											//"\r"
	return position;								//Rückgabe der Position in 1/100mm

}


//---------------------   Kreuzchen für Checkbox   ----------------//
//----------------           Tested: O.K.          ----------------//
void draw_cross(uint8_t y,uint8_t color)
{
	lcd_set_pixel(120,y,color);
	lcd_set_pixel(119,y,color);
	lcd_set_pixel(121,y,color);
	lcd_set_pixel(120,y-1,color);
	lcd_set_pixel(120,y+1,color);
}



