#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <stdint.h>
#include "global.h"
#include "keyboard.h"
#include "framebuffer.h"
#include "vt100.h"
#include "uart.h"

static volatile uint8_t g_vsync; // flag indicating begin of vertical sync period

//volatile register uint8_t char_l asm("r3");

#define GM_LOGO          0
#define GM_TERMINAL      1
#define GM_PRESENTATION  2

static uint8_t g_mode = GM_LOGO;

#define VST_SYNC   0
#define VST_BACK   1
#define VST_ACTIVE 2
#define VST_FRONT  3

//
// LOCAL FUNCTIONS
//
void init_logo(void);
void vsync_logo(void);
void vsync_terminal();
void vsync_presentation();
void vsync_handler();

//
// VSYNC ISR
//
// Activated by active video ISR at the end of every frame
//
ISR(TIMER2_COMPA_vect)
{
	static uint8_t state = VST_SYNC, next_state = VST_SYNC;

	switch(state)
	{
	case VST_SYNC:
		//VSYNC_PORT |= (1<<VSYNC_PIN);
		VSYNC_PORT &= ~(1<<VSYNC_PIN);         // vsync pulse is active-low  KEL 
      
		OCR2A = V_SYNC * H_TOTAL * 8 / 64;
		g_vsync = 1;
		next_state = VST_BACK;
		break;
	case VST_BACK:
		//VSYNC_PORT &= ~(1<<VSYNC_PIN);
		VSYNC_PORT |= (1<<VSYNC_PIN);         // vsync idles high  KEL 
      
		OCR2A = ((V_BACK) * H_TOTAL * 8 / 64) - 1;
		next_state = VST_ACTIVE;
		break;
	case VST_ACTIVE:
		g_vsync = 0;
		// deactivate myself
		TCCR2B &= ~(1<<CS22);
		OCR2A = V_FRONT * H_TOTAL * 8 / 64 - 1; // prepare for next invocation of this ISR
		TCNT2 = 0;
		// activate active video ISR
		TIFR1 = (1<<OCF1B);
		TIMSK1 |= (1<<OCIE1B);
		next_state = VST_SYNC;
		break;
	default: 
		state = next_state = VST_SYNC;
	}

	state = next_state;

}

//
// Active video ISR
//
// This function generates the whole video line
//
ISR(TIMER1_COMPB_vect)
{
	static uint16_t line_no = 0;
	static uint8_t *frame_p = (uint8_t*)framebuf;
	
	uint8_t char_l = line_no % 8;
	
	if(line_no == V_ACTIVE)
	{
		frame_p = (uint8_t*)framebuf;
		line_no = 0;
		TIMSK1 &= ~(1<<OCIE1B); // deactivate myself
		TCCR2B |= (1<<CS22); // activate VSYNC interrupt, prescaler=64

		return;
	}

	// For discussion of the inline assembly segment: 
	// http://www.avrfreaks.net/index.php?name=PNphpBB2&file=viewtopic&p=595112#595112

	uint8_t *temp_font;  // dummy variables - only needed to tell compiler that input variables will be changed
	uint8_t *temp_frame; // they should be discarded during optimization

  	asm volatile( 
        "add r31, %[char_l]"           "\n\t"
        "jmp 1f"                       "\n\t" 
        ".text 2"                      "\n\t" 
        "1:"                           "\n\t" 
        ".rept 80"                     "\n\t" 
        "ld r30, Y+"                   "\n\t" 
        "lpm __tmp_reg__, Z"           "\n\t" 
        "out %[color], __tmp_reg__"    "\n\t" 
        "out %[shift], __zero_reg__"   "\n\t" 
        "out %[shift], %[set_pl]"      "\n\t" 
        ".endr"                        "\n\t" 
        "jmp 2f"                       "\n\t" 
        ".text"                        "\n\t" 
        "2:"                           "\n\t" 
        :            "=y" (temp_frame),               // <-- output operands
                     "=z" (temp_font)
        : [shift]    "I" (_SFR_IO_ADDR(SHIFT_PORT)), 
          [color]    "I" (_SFR_IO_ADDR(COLOR_PORT)),  // <-- input operands
          [set_pl]   "r" (1<<SHIFT_PL),                
          [char_l]   "r" (char_l),
                     "y" (frame_p), 
                     "z" (font) 
    );

	if(char_l == 0x07)
		frame_p += H_CHARS;

	line_no++;

	// Check for characters from RS-232
	if(UCSR0A & (1<<RXC0))
	{
		uart0_buf[uart0_index++] = UDR0;
	}

	// Check for Characters from Keyboard
	if(UCSR1A & (1<<RXC1))
	{
		kb_buffer[kb_bufpos++] = UDR1;
	}
}

void vsync_blinkcursor()
{
	static uint8_t counter = 0;

	counter++;
	if(counter == 12)
	{
		counter = 0;

		cursor_blink();
	}
}


void vsync_logo()
{
	static uint8_t i = 0;
	static uint8_t t1 = 0;
	static uint8_t b_inc = 1;
	static uint8_t b_pos = 10;
	uint8_t x, y;
	static uint16_t sidescroll_start = 0;

	// Generate ASCII-marquee and boucing smiley animations
	t1++;
	if(t1 == 4) 
	{
		t1=0;
		
		// marquee
		uint8_t c = i++;
		for(x=10 ; x<70 ; x++)
		{
			fb_putchar_xy(x, 19, c);
			fb_putchar_xy(H_CHARS-1 - x, 21, c++);
		}

		// moving smiley
		fb_putchar_xy(b_pos, 25, ' ');
		b_pos += b_inc;
		fb_putchar_xy(b_pos, 25, 2);

		if(b_pos > 69) b_inc = -b_inc;
		if(b_pos < 10) b_inc = -b_inc;

		// side-scroll
		sidescroll_start++;
		if(sidescroll_start >= SIDESCROLL_ROWS) sidescroll_start = 0;
		for(y = 0 ; y < SIDESCROLL_COLS  ; y++)
		{
			for(x = 10 ; x < (H_CHARS-10)-1 ; x++)
			{
				fb_putchar_xy(x, y+32, fb_getchar_xy(x+1, y+32) );
			}
			uint8_t c = pgm_read_byte(&sidescroll[ (sidescroll_start*SIDESCROLL_COLS) + y ]);
			fb_putchar_xy(x, y + 32, c); 			
		}
	}
}

void vsync_terminal()
{
	uint8_t i;

	vsync_blinkcursor();

	// Display characters received from RS-232
	for(i=0 ; i<uart0_index ; i++)
	{
		vt100_handle_input(uart0_buf[i]);
	}
	uart0_index = 0;

	// decode and display keyboard input
	for(i=0 ; i<kb_bufpos ; i++)
	{
		decode(kb_buffer[i]);
	}
	kb_bufpos = 0;
}


#define KB_EXTENDED    0xE0
#define KB_ARROW_RIGHT 0x74
#define KB_ARROW_LEFT  0x6B 

void vsync_presentation()
{
	static uint8_t slide_no = 0;
	static uint8_t slide_old = 255;
	static uint8_t extended = 0;

	// ugly hack: since decoding of extended scancodes is not implemented in keyboard.c (yet)
	// do it locally... :-(

	for(uint8_t i=0 ; i<kb_bufpos ; i++)
	{
		uint8_t c = kb_buffer[i];

		if(c == KB_EXTENDED) 
			extended = 1;
		else if(extended && (c == KB_ARROW_RIGHT))
		{
			slide_no++;
			if(slide_no >= SLIDES_TOTAL) slide_no = SLIDES_TOTAL-1;
		}
		else if(extended && (c == KB_ARROW_LEFT))
		{
			slide_no--;
			if(slide_no >= SLIDES_TOTAL) slide_no = 0;
		}
		else 
		{
			extended = 0;

			if(c==0x2c) // 't'
			{
				g_mode = GM_TERMINAL;
				fb_clear();
				cursor_set(0, 0);
				return;
			}
			if(c==0x4b) // 'l'
			{
				g_mode = GM_LOGO;
				init_logo();
				cursor_set(0, 0);
				return;
			}
		}
	}
	kb_bufpos = 0;

	if(slide_no != slide_old)
	{
    	PGM_P p;

		fb_clear();
    	memcpy_P(&p, &slides[slide_no], sizeof(PGM_P));
	    strcpy_P((char *)framebuf, p);
 
		slide_old = slide_no;
	}

	fb_putstr_xy_P(10, V_CHARS-1, PSTR("Slide   /  "));
	fb_putchar_xy(16, V_CHARS-1, '1'+slide_no);
	fb_putchar_xy(20, V_CHARS-1, '0'+SLIDES_TOTAL);
}

void vsync_handler() // must NOT run in parallel with active video ISR!!!
{
	switch(g_mode)
	{
		case GM_LOGO:
			if(kb_bufpos && (kb_buffer[kb_bufpos-1]==0x29)) { // check for space key
				g_mode = GM_PRESENTATION;
				fb_clear();
				cursor_set(0, 0);
				kb_bufpos = 0;
				break;
			}
			kb_bufpos = 0;
			vsync_logo();
			break;
		case GM_TERMINAL:
			vsync_terminal();
			break;
		case GM_PRESENTATION:
			vsync_presentation();
			break;
	}
}

void init_logo()
{
	uint8_t y, c=0;
	
	fb_clear();

    for(y=18 ; y<V_CHARS ; y++)
	{
            framebuf[y*H_CHARS] = '0'+c;
            framebuf[y*H_CHARS + 79] = '0'+c;
			c++;
	}

	fb_putstr_xy_P(FB_CENTERED, 28, PSTR("Digitale Messdatenverarbeitung mit \x81" "bertakteten Mikrorechnern"));
	fb_putstr_xy_P(26, 45, PSTR("Und LPM ist doch n\x81" "tzlich!!!"));
	fb_putstr_xy_P(FB_CENTERED, 48, PSTR("Leertaste dr\x81" "cken zum Fortsetzen..."));

	strcpy_P((char *)framebuf, splash);
}

int main(void)
{
	init_logo();

    // initialize ports
    
    COLOR_DDR = 0xFF;
    COLOR_PORT = 0x00;

	SHIFT_DDR |= (1<<SHIFT_CE)|(1<<SHIFT_PL);
	SHIFT_PORT = (1<<SHIFT_PL);

    HSYNC_DDR |= (1<<HSYNC_PIN);
    VSYNC_DDR |= (1<<VSYNC_PIN);

	// Initialize Timer 1: HSYNC, active video
	// Timer 1 will call the active video interrupt function via OCR1B (isr.s)
	// and output the HSYNC pulse via OC1A

    TCCR1B = 0; 					// stop timer
	TCCR1B = (1<<WGM13)|(1<<WGM12); // Mode 14: Fast PWM counting up to ICR1
	TCCR1A = (1<<WGM11)|(1<<COM1A1)|(1<<COM1A0);// INVERTED PWM output
	TIMSK1 = (1<<OCIE1B);			// execute ISR on OC1B match
	ICR1   = H_TOTAL-1;				// timer period is one full scanline minus 1 for interrupt latency
	OCR1A  = H_SYNC-1;	
	// ISR start to output of first pixel: 51 cycles
	// wake up from idleand enter ISR: ~10 cycles     => OCR1B - 8
	// HSYNC to ACTIVE should be 3.19 us
	OCR1B  = H_SYNC + H_BACK - 9 - 1;	// substract some cycles for initialisation in ISR
	TCNT1 = 0;

	// Initialize Timer 2: VSYNC
	// Timer 2 will be enabled from active video ISR after end of active frame

	TCCR2B = 0;
	TCCR2A = (1<<WGM21); // Mode 2: CTC
	TIMSK2 = (1<<OCIE2A);
	TCNT2 = 0;
	OCR2A = V_FRONT;

	// Initialize UART0 (RS-232)
	// 9600 baud, 8n1
	//
	// 25 MHz:
	// --------------
	// UBRR0 = F_CPU / 16 / BAUD - 1 = 161.76 = 162
	// BAUD = F_CPU / 16 / (UBRR0 + 1) = 9585.89
	// error = BAUD / 9600 * 100% = 0.14%
	//
	// maximum transmission speed:
	// UART0_BUFLEN * video refresh rate
	// = 16 byte * 70 Hz = 1120 byte/sec = 8960 baud
	// ------------
	//
	// 20 MHz
	// UBRR0 = F_CPU / 16 / BAUD - 1 = 129
	// BAUD = F_CPU / 16 / (UBRR + 1) = 9615
	// error = baud / 9600 * 100% = 0.16%
	//
	UCSR0A = 0;
	UCSR0B = (1<<RXEN0)|(1<<TXEN0);
	UCSR0C = (1<<UCSZ01)|(1<<UCSZ00);
	UBRR0 = 162; // 25 MHz
	// UBRR0 = 129; // 20 MHz

	// Initialize UART1 (Keyboard)
	kbInit();

    // Start Timer 1
    sei();
    TCCR1B |= (1<<CS11); // prescaler = 8
	
	//
	// SLEEP mode is necessary to eliminate interrupt jitter:
	// due to processor instructions taking 2 or 3 clock cycles
	// ISRs can't always execute "on time".
	// When waking up from SLEEP modes always takes a CONSTANT
	// number of cycles --> problem solved
	//
	set_sleep_mode(SLEEP_MODE_IDLE);

	while(1)
	{
		sleep_enable();
        sei();
        sleep_cpu();
        sleep_disable();

		if(g_vsync)
		{
			vsync_handler();
			g_vsync = 0;
		}
	}
	
	return 0;
}
