//
// framebuffer.c
//
// Set of function that unify putting text in the framebuffer and
// allow management of a virtual cursor.
//
// (c) 2009 by Sebastian Brueckner <sb@thebackend.de>
//

#include <avr/io.h>
#include <stdint.h>
#include "global.h"
#include "framebuffer.h"

volatile uint8_t framebuf[H_CHARS*V_CHARS];

static volatile uint8_t cursor_x, cursor_y;
static volatile uint8_t cursor_x1, cursor_y1;
static volatile uint8_t cursor_x2, cursor_y2;

void cursor_advance()
{
	cursor_x++;
	if(cursor_x == H_CHARS)
	{
		cursor_y++;
		cursor_x = 0;
	}

	// TODO: implement scrolling
	if(cursor_y == V_CHARS)
	{
		fb_clear();
		cursor_y = 0;
	}
}

void cursor_newline()
{
	fb_putchar(0);

	cursor_y++;
	cursor_x = 0;
	if(cursor_y == V_CHARS)
	{
		fb_clear();
		cursor_y = 0;
	}
}

void cursor_set(uint8_t x, uint8_t y)
{
	cursor_x = x;
	cursor_y = y;
}

void cursor_push(uint8_t x, uint8_t y)
{
	cursor_x2 = cursor_x1;
	cursor_x1 = cursor_x;
	cursor_x = x;

	cursor_y2 = cursor_y1;
	cursor_y1 = cursor_y;
	cursor_y = y;
}

void cursor_pop()
{
	cursor_x = cursor_x1;
	cursor_x1 = cursor_x2;

	cursor_y = cursor_y1;
	cursor_y1 = cursor_y2;
}

void cursor_blink()
{
	static uint8_t c = 0; 
	static uint8_t state = 0;
	static uint8_t oldx, oldy;

	if(state)
	{
		if((oldx == cursor_x) && (oldy == cursor_y))
			fb_putchar_xy(oldx, oldy, c);
		state = 0;
	}
	else
	{
		c = fb_getchar();
		fb_putchar(2);

		// save cursor position
		oldx = cursor_x;
		oldy = cursor_y;

		state = 1;
	}

}

void fb_putchar(uint8_t c)
{
	framebuf[cursor_y * H_CHARS + cursor_x] = c;
}

uint8_t fb_getchar()
{
	return framebuf[cursor_y * H_CHARS + cursor_x];
}

/* moved to header!
void fb_putchar_xy( uint8_t x, uint8_t y, uint8_t c)
{
	framebuf[y * H_CHARS + x] = c;
}
*/

void fb_putstr_P(PGM_P str)
{
	uint8_t c, i=0;

	while((c = pgm_read_byte(&str[i++])))
	{
		fb_putchar(c);
		cursor_advance();
	}
}

void fb_putstr_xy_P(uint8_t x, uint8_t y, PGM_P str)
{
	uint8_t c, i=0;
	
	cursor_push(0, 0);
	
	if(x == FB_CENTERED)
	{
		x = (H_CHARS - strlen_P(str)) / 2;
	}
	
	cursor_set(x, y);

	while((c = pgm_read_byte(&str[i++])))
	{
		fb_putchar(c);
		cursor_advance();
	}
	
	cursor_pop();
}

void fb_clear()
{
	uint16_t i;
	for(i=0 ; i<(V_CHARS*H_CHARS) ; i++)
	{
		framebuf[i] = 0;
	}
}
