///////////////////////////////////////////////////////////////////////
//                        ags BINARY CLOCK                           //
///////////////////////////////////////////////////////////////////////
//
// (c) Martin Wenske <martinw@ags.tu-bs.de>
//
// Intial Release: 2011-01-21
// Last Change:
// - portability increased
//
// Revision 0.3:
//  - Bugfix | portability increased
// Revision 0.2:
//  - Bugfix | minute length corrected
// Revision 0.1:
//  - Initial release
// ====================================================
// Size after:
// AVR Memory Usage
// ----------------
// Device: attiny2313
//
// Program:    1046 bytes (51.1% Full)
// (.text + .data + .bootloader)
// 
// Data:         13 bytes (10.2% Full)
// (.data + .bss + .noinit)

#ifndef F_CPU
#define F_CPU  8000000
#endif

#ifndef __AVR_ATtiny2313__
#define __AVR_ATtiny2313__
#endif


/**************************************************/
/****************** Includes **********************/ 
/**************************************************/

#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>

/**************************************************/
/**************** Lokal Defines *******************/ 
/**************************************************/

#define FALSE 0
#define TRUE 1
                        
// Hours: 5 LEDs

#define LED_HOUR1_DDR   DDRB
#define LED_HOUR1_PORT  PORTB
#define LED_HOUR1       PORTB2

#define LED_HOUR2_DDR   DDRB
#define LED_HOUR2_PORT  PORTB
#define LED_HOUR2       PORTB0

#define LED_HOUR3_DDR   DDRD
#define LED_HOUR3_PORT  PORTD
#define LED_HOUR3       PORTD5

#define LED_HOUR4_DDR   DDRD
#define LED_HOUR4_PORT  PORTD
#define LED_HOUR4       PORTD3

#define LED_HOUR5_DDR   DDRD
#define LED_HOUR5_PORT  PORTD
#define LED_HOUR5       PORTD1

// Minutes: 6 LEDs
 
#define LED_MIN1_DDR  DDRB
#define LED_MIN1_PORT PORTB
#define LED_MIN1      PORTB3
 
#define LED_MIN2_DDR  DDRB
#define LED_MIN2_PORT PORTB
#define LED_MIN2      PORTB1
 
#define LED_MIN3_DDR  DDRD
#define LED_MIN3_PORT PORTD
#define LED_MIN3      PORTD6
 
#define LED_MIN4_DDR  DDRD
#define LED_MIN4_PORT PORTD
#define LED_MIN4      PORTD4
 
#define LED_MIN5_DDR  DDRD
#define LED_MIN5_PORT PORTD
#define LED_MIN5      PORTD2
 
#define LED_MIN6_DDR  DDRD
#define LED_MIN6_PORT PORTD
#define LED_MIN6      PORTD0


// Push buttons
#define BUTTON_B_DDR    DDRB
#define BUTTON_B_PORT   PORTB
#define BUTTON_B_PIN    PINB
#define BUTTON_B        PB4

#define BUTTON_A_DDR    DDRB
#define BUTTON_A_PORT   PORTB
#define BUTTON_A_PIN    PINB
#define BUTTON_A        PB5

/**************************************************/
/*************** Global Defines *******************/ 
/**************************************************/

volatile uint32_t sys_tick = 0; // luft alle 139,63558884727424053266749895963 Jahre ueber ;-)
volatile uint32_t sys_tick_alt = 0;
volatile uint8_t min_count = 0;
volatile uint8_t hour_count = 0;
volatile uint8_t set_time_flag = FALSE;

/**************************************************/
/**************** Lokal Defines *******************/ 
/**************************************************/

uint8_t led_to_change = 0;
uint8_t start_flag = FALSE;

/**************************************************/
/*************** Lokal Funktions ******************/ 
/**************************************************/
void led_init(void)
{                                
  LED_MIN1_DDR |= (1<<LED_MIN1);
  LED_MIN2_DDR |= (1<<LED_MIN2);
  LED_MIN3_DDR |= (1<<LED_MIN3);
  LED_MIN4_DDR |= (1<<LED_MIN4);
  LED_MIN5_DDR |= (1<<LED_MIN5);
  LED_MIN6_DDR |= (1<<LED_MIN6);
                                   
  LED_HOUR1_DDR |= (1<<LED_HOUR1);
  LED_HOUR2_DDR |= (1<<LED_HOUR2);
  LED_HOUR3_DDR |= (1<<LED_HOUR3);
  LED_HOUR4_DDR |= (1<<LED_HOUR4);
  LED_HOUR5_DDR |= (1<<LED_HOUR5);
}

void button_init(void)
{                                                           
  BUTTON_A_PORT |= (1<<BUTTON_A);
  BUTTON_B_PORT |= (1<<BUTTON_B);

  BUTTON_A_DDR &= ~(1<<BUTTON_A);
  BUTTON_B_DDR &= ~(1<<BUTTON_B);
} 

void irq_init(void)
{
	TCCR1B |= (1<<WGM12) | (1<<CS12);      // prescaler = 256
	OCR1A = 31249; // 8.000.000 / 256 - 1 = 31.249,0
	TCNT1 = 0;
	TIMSK |= (1<<OCIE1A); // enable compare match ISR
	sei();
}
                    
void set_leds(void)
{
  if( (sys_tick - sys_tick_alt) >= 60 ) 
  {
    sys_tick_alt = sys_tick; 
    min_count++;
  }

  if(min_count >= 60) 
  {
    min_count = 0;
    hour_count++;
  }
  
  if(hour_count >= 24)
  {
    hour_count = 0;
  }
                                  
   if(min_count & 1)  LED_MIN1_PORT |= (1<<LED_MIN1);
  else LED_MIN1_PORT &= ~(1<<LED_MIN1);
  if(min_count & 2)  LED_MIN2_PORT |= (1<<LED_MIN2);
  else LED_MIN2_PORT &= ~(1<<LED_MIN2);       
  if(min_count & 4)  LED_MIN3_PORT |= (1<<LED_MIN3);
  else LED_MIN3_PORT &= ~(1<<LED_MIN3);       
  if(min_count & 8)  LED_MIN4_PORT |= (1<<LED_MIN4);
  else LED_MIN4_PORT &= ~(1<<LED_MIN4);       
  if(min_count & 16)  LED_MIN5_PORT |= (1<<LED_MIN5);
  else LED_MIN5_PORT &= ~(1<<LED_MIN5);       
  if(min_count & 32)  LED_MIN6_PORT |= (1<<LED_MIN6);
  else LED_MIN6_PORT &= ~(1<<LED_MIN6); 
                                                       
  if(hour_count & 1)  LED_HOUR1_PORT |= (1<<LED_HOUR1);
  else LED_HOUR1_PORT &= ~(1<<LED_HOUR1); 
  if(hour_count & 2)  LED_HOUR2_PORT |= (1<<LED_HOUR2);
  else LED_HOUR2_PORT &= ~(1<<LED_HOUR2); 
  if(hour_count & 4)  LED_HOUR3_PORT |= (1<<LED_HOUR3);
  else LED_HOUR3_PORT &= ~(1<<LED_HOUR3); 
  if(hour_count & 8)  LED_HOUR4_PORT |= (1<<LED_HOUR4);
  else LED_HOUR4_PORT &= ~(1<<LED_HOUR4); 
  if(hour_count & 16)  LED_HOUR5_PORT |= (1<<LED_HOUR5);
  else LED_HOUR5_PORT &= ~(1<<LED_HOUR5); 
} 

void leds_off(void)
{
  LED_MIN1_PORT &= ~(1<<LED_MIN1);
  LED_MIN2_PORT &= ~(1<<LED_MIN2);       
  LED_MIN3_PORT &= ~(1<<LED_MIN3);       
  LED_MIN4_PORT &= ~(1<<LED_MIN4);       
  LED_MIN5_PORT &= ~(1<<LED_MIN5);       
  LED_MIN6_PORT &= ~(1<<LED_MIN6);

  LED_HOUR1_PORT &= ~(1<<LED_HOUR1); 
  LED_HOUR2_PORT &= ~(1<<LED_HOUR2); 
  LED_HOUR3_PORT &= ~(1<<LED_HOUR3); 
  LED_HOUR4_PORT &= ~(1<<LED_HOUR4); 
  LED_HOUR5_PORT &= ~(1<<LED_HOUR5);
}

void do_start_sequenz(void)
{  
  leds_off();
  LED_MIN1_PORT |= (1<<LED_MIN1);
  _delay_ms(100);
  leds_off();
  LED_MIN2_PORT |= (1<<LED_MIN2);
  _delay_ms(100); 
  leds_off();
  LED_MIN3_PORT |= (1<<LED_MIN3);
  _delay_ms(100); 
  leds_off();
  LED_MIN4_PORT |= (1<<LED_MIN4);
  _delay_ms(100);
  leds_off();
  LED_MIN5_PORT |= (1<<LED_MIN5);
  _delay_ms(100);
  leds_off();
  LED_MIN6_PORT |= (1<<LED_MIN6);
  _delay_ms(100);
  leds_off();
                          
  LED_HOUR1_PORT |= (1<<LED_HOUR1);
  _delay_ms(100);
  leds_off();
  LED_HOUR2_PORT |= (1<<LED_HOUR2);
  _delay_ms(100); 
  leds_off();
  LED_HOUR3_PORT |= (1<<LED_HOUR3);
  _delay_ms(100); 
  leds_off();
  LED_HOUR4_PORT |= (1<<LED_HOUR4);
  _delay_ms(100);
  leds_off();
  LED_HOUR5_PORT |= (1<<LED_HOUR5);
  _delay_ms(100);
  leds_off();
}
 
int main(void)
{ 
  led_init();
  button_init();
  do_start_sequenz();

  while(1)
  {
    if(set_time_flag && start_flag)
    {
      set_leds();
      set_time_flag = FALSE;
    }

    while(!start_flag)
    {
      if(!(BUTTON_A_PIN & (1<<BUTTON_A))) // led wechseln
      {
        _delay_ms(300);
        led_to_change++;
        if(led_to_change > 10) led_to_change = 0;

        while(!(BUTTON_A_PIN & (1<<BUTTON_A)))
        {
          if((!(BUTTON_B_PIN & (1<<BUTTON_B))) && !start_flag)
          {
            _delay_ms(300);
            do_start_sequenz();
            irq_init();
            start_flag = TRUE;  // flag fr Programmstart setzen
          }
        }
      }

      if(!(BUTTON_B_PIN & (1<<BUTTON_B)) && !start_flag) // led an/aus machen
      {
        _delay_ms(300);
        if(led_to_change <= 5) min_count ^= (1<<led_to_change); // change min counter
        if(led_to_change > 5 && led_to_change <= 10) hour_count ^= (1<<(led_to_change-6));  // change hour counter

        while(!(BUTTON_B_PIN & (1<<BUTTON_B)))
        {
          if((!(BUTTON_A_PIN & (1<<BUTTON_A))) && !start_flag)
          {
            _delay_ms(300);
            do_start_sequenz();
            irq_init();
            start_flag = TRUE;  // flag fr Programmstart setzen
          }
        }
      }
      set_leds();
    }
  }

}

/**************************************************/
/********** Interrupt Service Routine *************/ 
/**************************************************/

ISR(TIMER1_COMPA_vect)
{
   sys_tick++;
   set_time_flag = TRUE;
}
